// 10 Feb 2022.
//
// sort table wsSurvey javascript library.
//
//   wsurvey.sortTable.init : initialize an existing table -- add "sort buttons" to the top (header cell of a column)
//   wsurvey.sortTable.status  : return an array of messages describing the sort status of the table
//   wsurvey.sortTable.highlight  : highlight the header cells of sorted columns (using bgcolor)
//   wsurvey.sortTable.rowColors(aid,rowColors)  :: setup and add a class that yields alternating row colors
//    wsurvey.sortTable.recentSortInfo(athis,aid)  ::return a string of descriptoin information on what this sort button (that was clicked) did (or would) do
//   wsurvey.sortTable.copyHeaderRow(aid,targetId) ::   copy the header row to a container not in the table
//   wsurvey.sortTable.unShrinkAll(aid) :: unsrhink shrunken columns
//    wsurvey.sortTable.markGroups : mark "contiguous groups " (cells next to each other with same value) with a cell border color
//   wsurvey.sortTable.setColWidths   : set the column widths
//   wsurvey.sortTable.attr  : read an attribute (checks for several variances)
 //
//  wsurvey.sortTable.sortCol : not typically used  -- it is what the "sort buttons" call. Maybe useful if you want to bypass wsurvey.sortTable.init
//
// * Quick useage:
//     wsurvey.sortTable.init('tableId')
//  will add "sort buttons" to all columns of a table.
// * You can customize by calling
//     wsurvey.sortTable.init('tableId',options)
//   where options is an associative array with a number of options
// * You can  add "data-_sortUse" attributes to cells (or to an element within a cell) for use when sorting (instead of using the acutal
//   contents of a cell
// * You can specify options (over riding the options in the options argument) on a column specific basis by including
//   attributes in the header column
//
// See wsurvey.sortTable.txt for further descriptions


//==========================
// Initialize wsurvey.sortTable (add buttons to top row)
// aid is the id of the table. Or jquery element of dom element.
// see wsurvey_sortTable.txt for a descriptoin of options. The most improtant ones are ascIcon and desIcon


if (typeof(window['wsurvey'])=='undefined') window['wsurvey']={};
wsurvey.sortTable={};

wsurvey.sortTable.init=function(aid,options) {

  if (arguments.length<2 || typeof(options)!='object') options={};
  var optsUse={},aidSay='n.a.', gotAid=0;
  let erows,eTds;

  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     aidSay=aid;
     gotAid=1;
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let etable=jQuery(aid);
  if (etable.length==0 ) return 'No such element (id='+aidSay+') ';
  if (gotAid==0) aidSay=etable.wsurvey_attr('id','n.a.');
  let atag=etable.prop('tagName');
  if (atag.toLowerCase()!='table') return 'No such element, or not a table (id='+aidSay+') ';
  erows=etable.find('tr');
  if (erows.length==0) return '-1 No TR elements in table = '+aidSay;


// the defaults (can be over ridded by stuff in options
  let defOpts={'sortAsc':1, 'sortNumeric':-1,                    // -1 means auto determine sort type
              'iconCurrentClass':'wsurvey_sortTable_iconCurrentClass',
              'icon2CurrentClass':'wsurvey_sortTable_icon2CurrentClass',
              'iconClass':'wsurvey_sortTable_iconClass',
              'sortUse':'_sortUse',  'headerRow':0,
              'ascIcon':'&uarr;',  'desIcon':'&darr;',
              'shrinkColIcon':'','rowNumberSortableIcon':'&Oscr;#',
              'asc2Icon':'',  'des2Icon':'',             // '' means "do NOT display"
              'startRow':'1', 'endRow':'' ,                 // '' means include the last row
              'rowNumberClass':'0',
              'rowNumberSortableClass':'0',
              'skipCols':[],'freezeCols':[],'preCall':[],'postCall':[],
              'addWidth':0
             };

   let customOpts=['sortNumeric','sortAsc','iconClass', 'ascIcon','desIcon','asc2Icon','des2Icon','shrinkColIcon'];  // can set on col specific basis (using attributes in headerrow td
   let optsFixems={'iconClass':'wsurvey_sortTable_iconClass',   // fix some "1" and "0" defaults
                     'ascIcon':'&uarr;', 'desIcon':'&darr;',
                     'asc2Icon':'&Uarr;','des2Icon':'&Darr;',
                     'shrinkColIcon':'&#9003;'};

   useOpts=sortTable_init_options(defOpts,options,customOpts);  // ----- set and double check the options -- start with defopts, replace it with anythhing specified in options!

   if (typeof(useOpts)=='string') return useOpts ;      // an error

// transform these global vaars (i.e.; convert '1' to a default icon

   var rowNumberClass,rowNumberSortableClass,freezeCols,freezeColSay,skipCols,startRow,endRow,headerRow,shrinkColIcon ;  // these are set in wsurvey.sortTable.init_options2

   eTds=sortTable_init_options2(erows,useOpts)   ; // eTds is "header row" td,th
   if (typeof(eTds)=='string') return eTds ;  // error occurred (should be jquery collection)

   useOpts['shrinkColIcon']=shrinkColIcon;   // used below

// add row number column?
  erow1=jQuery(erows[headerRow]);  // header row stuff might change ....
  if (rowNumberSortableClass!='') {
    erows= sortTable_init_addRowNumbers(rowNumberSortableClass,erows,1,headerRow, useOpts['rowNumberSortableIcon']);
  }
  if (rowNumberClass!='') {  // fixed row number should go before  sortable (prepend  is used
    erows= sortTable_init_addRowNumbers(rowNumberClass,erows,0,headerRow,'#');
  }
  let erowHeader=jQuery(erows[headerRow]);   // so re read it here
  eTds=erowHeader.find('th,td');

  erowHeader.attr('data-wsurveysorttable_headerrow',headerRow);  // existence of this attribute signals this is the header row


// save attributes to table (used by wsurvey.sortTable.sortCol)
  etable.attr('data-wsurveysorttable_init',1);
  etable.attr('data-wsurveysorttable_startrow',startRow);
  etable.attr('data-wsurveysorttable_endrow',endRow);

// Note: saving as attributes (hence the use of csvs) facilitates the standalone use of  wsurvey.sortTable.sortCol
  etable.attr('data-wsurveysorttable_freezecols',freezeColSay);
  etable.attr('data-wsurveysorttable_precalls',useOpts['preCall'].join(','));
  etable.attr('data-wsurveysorttable_postcalls',useOpts['postCall'].join(','));

// for display purposes (used by wsurvey.sortTable.status)
  let colDescs=[];
  let  colTextTotal=0;
  for (let itt=0;itt<eTds.length;itt++) {
     let vetd=jQuery(eTds[itt]);
     vdesc=vetd.wsurvey_attr('data-wsurveysorttable_desc',false,1);
     if (vdesc===false) vdesc=vetd.text();
     colDescs[itt]=jQuery.trim(vdesc);
    let  colTextTotal=0;
  }

// save as .data, since not used by   wsurvey.sortTable.sortCol
  etable.data('wsurvey_sortTable_colDescs',colDescs);

// these are "shortcuts" that  wsurvey.sortTable.sortCol uses to avoid lookups and qc checks
  etable.data('wsurvey_sortTable_init',1);
  etable.data('wsurvey_sortTable_startRowUse',startRow);
  etable.data('wsurvey_sortTable_endRowUse',endRow);
  etable.data('wsurvey_sortTable_sortUse',useOpts['sortUse']);
  etable.data('wsurvey_sortTable_freezeColsUse',freezeCols);
  etable.data('wsurvey_sortTable_erowHeader',erowHeader);
  etable.data('wsurvey_sortTable_headerrow',headerRow);

  var ndid=0;

//  ::: For each colum nf of the table ....  create the "sort buttons", and add to the header cells

  var  madeButton=0;

  for (let ie=0;ie<eTds.length;ie++) {           // for each of the Header row cells.... add buttons! ...................

    if (jQuery.inArray(ie,skipCols)>-1 || jQuery.inArray(ie,freezeCols)>-1)    continue ;  // skip this column

    let ieOrig= (rowNumberClass=='') ? ie : ie-1 ;
    ieOrig= (rowNumberSortableClass=='') ? ieOrig : ieOrig-1 ;

    let optsUseHere={};     // copy (not a clone)!
    for (let ie2 in useOpts) optsUseHere[ie2]=useOpts[ie2];

// see if any of the options are defined in the <th> or <td>. If not, use the options (or perhaps the default options
// customOpts=['sortNumeric','sortAsc','iconClass', 'ascIcon','desIcon','asc2Icon','des2Icon','shrinkColIcon'];  // can set on col specific basis (using attributes in headerrow td

    etd1=jQuery(eTds[ie]);      // look for in-table overrides of  options
    optsUseHere=sortTable_init_options3(etd1,useOpts,customOpts,optsFixems) ;
    if (optsUseHere['ascIcon']=='' && optsUseHere['desIcon']==''  &&  optsUseHere['asc2Icon']=='' &&  optsUseHere['des2Icon']=='') continue ; // suppress sorting, so skip

    madeButton++;
    var buttonText="??";
    let acoltext=colDescs[ie];
    acoltext = jQuery("<div>").html(acoltext).text();  // remove html
    acoltext = acoltext.replace(/\"/g, '');   // remove "  chaacters

    buttonString='<button name="wsurvey_sortTable_button"  ';
    buttonString+=' data-wsurveysorttable_madebutton="'+madeButton+'"  ';
    buttonString+='    data-wsurveysorttable_origcolnumber="'+ieOrig+'"   ';
    buttonString+='    data-wsurveysorttable_colnumber="'+ie+'"  ';
    buttonString+='    data-wsurveysorttable_sortsecondary="0" ';
    buttonString+='    data-wsurveysorttable_ascicon="'+optsUseHere['ascIcon']+'" ';
    buttonString+='    data-wsurveysorttable_desicon="'+optsUseHere['desIcon']+'"   ';
    buttonString+='    data-wsurveysorttable_sortnumeric="'+jQuery.trim(optsUseHere['sortNumeric'])+'" ';
    if (jQuery.trim(optsUseHere['iconClass'])!='') buttonString +=' class="'+optsUseHere['iconClass']+'" ';
    if (jQuery.trim(optsUseHere['iconCurrentClass'])!='') {
           buttonString +=' data-wsurveysorttable_iconcurrentclass="'+optsUseHere['iconCurrentClass']+'"   ';
    }

    if (jQuery.trim(optsUseHere['sortAsc'])==1) {
       buttonText=optsUseHere['ascIcon']
       buttonString+=' value="'+acoltext+'" data-wsurveysorttable_sortasc="1"  title="sort ascending" ';
    } else {
       buttonText=optsUseHere['desIcon']
       buttonString+=' value="'+acoltext+'" data-wsurveysorttable_sortasc="0"  title="sort descending"  ';
    }
    buttonString+='>'+buttonText+'</button>\n';

    if (optsUseHere['asc2Icon']!='' ||  optsUseHere['des2Icon']!='') {        // add 2ndary sort buttons
        madeButton++;
        buttonString+=' <button  name="wsurvey_sortTable_button"   ';
        buttonString+='          data-wsurveysorttable_madebutton="'+madeButton+'" ';
        buttonString+='          data-wsurveysorttable_origcolnumber="'+ieOrig+'" ';
        buttonString+='          data-wsurveysorttable_colnumber="'+ie+'" ';
        buttonString+='          data-wsurveysorttable_sortsecondary="1" ';
        buttonString+='    data-wsurveysorttable_ascicon="'+optsUseHere['asc2Icon']+'" ';
        buttonString+='    data-wsurveysorttable_desicon="'+optsUseHere['des2Icon']+'"   ';
        buttonString+='   data-wsurveysorttable_sortnumeric="'+jQuery.trim(optsUseHere['sortNumeric'])+'" ';
        if (jQuery.trim(optsUseHere['icon2CurrentClass'])!='') {
           buttonString +=' data-wsurveysorttable_iconCurrentClass="'+optsUseHere['icon2CurrentClass']+'"   '; // note that iconcurrentclass is used for both primary and secondayr buttos
        }
        if (jQuery.trim(optsUseHere['iconClass'])!='') buttonString +=' class="'+optsUseHere['iconClass']+'" ';
        if (jQuery.trim(optsUseHere['sortAsc'])==1) {
          buttonText=optsUseHere['asc2Icon']
          buttonString+=' value="2ndary ascend" sortAsc="1"  title="sort ascending (secondary) " ';
        } else {
          buttonText=optsUseHere['des2Icon']
          buttonString+=' value="2ndary descend" sortAsc="0"  title="sort descending  (secondary)"  ';
        }
        buttonString+=' >'+ buttonText+'</button>';
    }
     if (optsUseHere['shrinkColIcon']!=''    ) {    // add shrink column button
        buttonString+=' <button class="'+optsUseHere['iconClass']+'" + title="Shrink this column" style="cursor:col-resize" name="wsurvey_sortTable_shrinkButton" ';
        buttonString+='      data-wsurveysorttable_origColNumber="'+ieOrig+'" ';
        buttonString+='      data-wsurveysorttable_colNumber="'+ie+'" ';
        buttonString+='      data-wsurveysorttable_isShrunk="0"  ';
        buttonString+='      data-wsurveysorttable_desc="'+acoltext+'" ';
//        shrinkColIcon="'+shrinkColIcon+'" ';
        buttonString+=' >'+ optsUseHere['shrinkColIcon']+'</button>';
     }

     let afoo=etd1.prepend(buttonString) ;

     let afoo2=etd1.find('[name="wsurvey_sortTable_button"]');
     afoo2.on('click',wsurvey.sortTable.sortCol);

     if (optsUseHere['shrinkColIcon']!=''   ) {
        let afoo3=etd1.find('[name="wsurvey_sortTable_shrinkButton"]');
        afoo3.on('click',{'useclass':'wsurvey_sortTable_shrinkClass'},wsurvey.sortTable.shrinkCol );
     }

     ndid++ ;

// assign the click event

  }     // ie


  let tfixed=etable.css('table-layout');
  let tableFixed =  (typeof(tfixed)!=='undefined' && tfixed=='fixed') ? 1 : 0 ;

// width of columns
  let tableWidth=etable.width();
  let tdWidths=[],tdScales=[];
  let tdWidthsTotal=0;
  for (let ie=0;ie<eTds.length;ie++) {
    let etd1=jQuery(eTds[ie]);      // look for in-table overrides of  options
    let twidth=etd1.width();
    tdWidthsTotal+=twidth;
    tdWidths[ie]=twidth
    tdScales[ie]=1.0;
    etd1.css({'width':twidth});   // make room for the arrows (so that headerCopy aligns
  }
  tdWidthsPct=[];
  for (let jt=0;jt<tdWidths.length;jt++) tdWidthsPct[jt]=tdWidths[jt]/tdWidthsTotal;
  etable.data('wsurvey_sortTable_colWidths',tdWidths);
  etable.data('wsurvey_sortTable_colWidthsTotal',tdWidthsTotal);
  etable.data('wsurvey_sortTable_colWidthsPct',tdWidthsPct);
  etable.data('wsurvey_sortTable_colScales',tdScales);

  etable.data('wsurvey_sortTable_fixed',tableFixed);
  etable.data('wsurvey_sortTable_tableWidth',tableWidth);

// create some default css classes
  let darules={};
  darules['wsurvey_sortTable_iconClass']={'border-radius':'8px','padding':'3px','margin':'0px','cursor':'n-resize',
         'border-bottom':'4px groove #aacccc','border-right':'5px groove #bbccbb','color':'blue'};
  darules['wsurvey_sortTable_iconCurrentClass']={'background-color':'#dbcbeb','border':'3px dotted blue'};
//  darules['wsurvey_sortTable_iconCurrentClass']={'background-color':'pink','border':'3px dotted blue'};
  darules['wsurvey_sortTable_icon2CurrentClass']={'background-color':'#dbebeb','border':'2px dotted brown'};
  darules['wsurvey_sortTable_rowNumberClass']={'background-color':'#cfcfcf','font-size':'75%','padding':'3px','opacity':'.7','font-style':'oblique'};
  darules['wsurvey_sortTable_rowNumberSortableClass']={'background-color':'#efefef','font-size':'90%','opacity':'.9','xfont-style':'oblique'};
  darules['wsurvey_sortTable_shrinkClass']={'padding':'0px','margin':'0px','background-color':'gray', 'overflow':'hidden  !important','font-size':'11%','opacity':'.4'};
  darules['wsurvey_sortSurvey_tempHide']={'display':'none !important'};

  wsurvey.addCssRule('wsurvey_sortTable_rowCss',darules);

 return ndid ;

// === several internal  (within  wsurvey.sortTable.init) are below...

// ================ ===================
// read and fix up options

function sortTable_init_options(defOpts,options,custompOpts)  { // sets "local" varriables

  var useOpts=defOpts ;   // useOPts is backfilled with stuff from options

// wsurvey.findObjectDefault=function(aobj,aindex,adef,notExact,all)

  wsurvey.updateObject(useOpts,options,0);   // update value in useOpts (using stuff specified in options arghument

// double check useOPts

  if (typeof(useOpts['skipCols'])!='object')  {  // make string into array
      let aopt=useOpts['skipCols'];
      useOpts['skipCols']=[aopt] ;
  }

  if (typeof(useOpts['freezeCols'])!='object')  { // make string into array
      let aopt=useOpts['freezeCols'];
      useOpts['freezeCols']=[aopt] ;
  }
  if (typeof(useOpts['preCall'])!='object' )  {   // make string into array
      let aopt=useOpts['preCall'];
      useOpts['preCall']=[aopt] ;
  }
  if (typeof(useOpts['postCall'])!='object' )  {   // make string into array
      let aopt=useOpts['postCall'];
      useOpts['postCall']=[aopt] ;
   }

// make sure precall functions exists
  for (iz=0;iz<useOpts['preCall'].length;iz++) {
       let afunc=jQuery.trim(useOpts['preCall'][iz]);
       if (typeof(afunc)=='string' && jQuery.trim(afunc)=='') continue;
       if (typeof(afunc)!=='string' ||  typeof(window[afunc])!='function') return '-1 An entry  ('+afunc+') in preCall is not a function ';
  }
// make sure postall functions exists
  for (iz=0;iz<useOpts['postCall'].length;iz++) {
       let afunc=jQuery.trim(useOpts['postCall'][iz]);
       if (typeof(afunc)=='string' && jQuery.trim(afunc)=='') continue;
       if (typeof(afunc)!=='string' ||  typeof(window[afunc])!='function') return '-1 An entry  ('+afunc+') in postCall is not a function ';
  }


// check for miscodings in options
  if (typeof(useOpts['ascIcon'])!='string')  useOpts['ascIcon'] ='' ;
  if (typeof(useOpts['asc2Icon'])!='string')  useOpts['asc2Icon'] ='' ;
  if (typeof(useOpts['desIcon'])!='string')  useOpts['desIcon'] ='' ;
  if (typeof(useOpts['des2Icon'])!='string')  useOpts['des2Icon'] ='' ;
  if (typeof(useOpts['iconCurrentClass'])!='string')   useOpts['iconCurrentClass'] ='' ;
  if (typeof(useOpts['icon2CurrentClass'])!='string')   useOpts['icon2CurrentClass'] ='' ;
  if (typeof(useOpts['iconClass'])!='string')  useOpts['iconClass'] ='' ;
  if (typeof(useOpts['rowNumberClass'])!='string' &&  typeof(useOpts['rowNumberClass'])!='number')   useOpts['rowNumberClass'] ='' ;
  if (typeof(useOpts['rowNumberSortableClass'])!='string' &&  typeof(useOpts['rowNumberSortableClass'])!='number')   useOpts['rowNumberSortableClass'] ='' ;
  if (typeof(useOpts['sortNumeric'])!='string' && typeof(useOpts['sortNumeric'])!='number')  useOpts['sortNumeric'] ='-1' ;  //0:alphabetic, 1:numeric, -1:auto (numeric cmparing numbers, otherwise alphabetic
  if (typeof(useOpts['headerRow'])!='string' && typeof(useOpts['headerRow'])!='number' )  useOpts['headerRow'] ='0' ;  //0:alphabetic, 1:numeric, -1:auto (numeric cmparing numbers, otherwise alphabetic
  if (typeof(useOpts['shrinkColIcon'])!='string' &&  typeof(useOpts['shrinkColIcon'])!='number')   useOpts['shrinkColIcon'] ='' ;
  if (typeof(useOpts['rowNumberSortableIcon'])!='string' ||  jQuery.trim(useOpts['rowNumberSortableIcon'])=='1')   useOpts['rowNumberSortableIcon'] ='&Oscr;#' ; // the default

   return useOpts ;
 }               // s_sortTable_init_options  (within  wsurvey.sortTable.init


//=========================
// fix up the options
// this chnges these globals:
//    rowNumberClass,rowNumberSortableClass,freezeCols,freezeColSay,skipCols,startRow,endRow,headerRow,shrinkColIcon,,des2Icon;

  function sortTable_init_options2(erows,useOpts) {
   rowNumberClass=jQuery.trim(useOpts['rowNumberClass']);
   if (rowNumberClass =='0')   rowNumberClass ='' ;
   if (rowNumberClass =='1')   rowNumberClass ='wsurvey_sortTable_rowNumberClass' ;  // hard coded class

   rowNumberSortableClass=jQuery.trim(useOpts['rowNumberSortableClass']);
   if (rowNumberSortableClass =='0')   rowNumberSortableClass ='' ;
   if (rowNumberSortableClass =='1')   rowNumberSortableClass ='wsurvey_sortTable_rowNumberSortableClass' ;  // hard coded class


   shrinkColIcon=jQuery.trim(useOpts['shrinkColIcon']);
   if (jQuery.trim(shrinkColIcon) =='0')   shrinkColIcon ='' ;
   if (shrinkColIcon =='1')   shrinkColIcon ='&#9003;' ;  // default icon
   shrinkColIcon=wsurvey.sortTable.html_entity_decode(shrinkColIcon);


  let colsAdd=0;         // the number of added columns (that user should NOT account for when specifyin frozen and skipped (non sortable) columns
  if (rowNumberClass!='') colsAdd++;
  if (rowNumberSortableClass!='') colsAdd++;

  freezeCols=useOpts['freezeCols'] ;
  if (colsAdd>0) {
     for (let iuu=0;iuu<freezeCols.length;iuu++) freezeCols[iuu]=freezeCols[iuu]+colsAdd  ;   // augment to deal with added columns
  }
  if (rowNumberClass!='') {         // augment to account for this additional (frozen) lead column
     freezeCols.unshift(0);  // rownumber column is frozen (its always col 0 (left most)
  }
  freezeColSay='';  // not needed , but useful for debugging
  if (freezeCols.length>0) freezeColSay=freezeCols.join(',');   // this is saved at a table attribure

  skipCols=useOpts['skipCols'];
  if (colsAdd>0) {
     for (let iuu=0;iuu<skipCols.length;iuu++) skipCols[iuu]=skipCols[iuu]+colsAdd ;   // augment to deal with added rowNumber column
  }


// make sure startrow, endrow, and headerrow are within appropriate range
    startRow=1;
  if (!isNaN(useOpts['startRow']) && jQuery.trim(useOpts['startRow'])!==''  && useOpts['startRow']>-1 )  startRow=parseInt(useOpts['startRow']);
  startRow=Math.max(0,startRow);

   endRow=erows.length-1 ;
  if (!isNaN(useOpts['endRow']) && jQuery.trim(useOpts['endRow'])!==''   )  endRow=parseInt(useOpts['endRow']);
  if (endRow<=0 ) endRow=(erows.length-1)+endRow      ;; // from end
  endRow=Math.max(startRow,endRow);
  endRow=Math.min(endRow,erows.length-1);

  headerRow=0;
  if (!isNaN(useOpts['headerRow']) && jQuery.trim(useOpts['headerRow'])!==''   ) headerRow=parseInt(useOpts['headerRow']) ;
  if (headerRow<0) headerRow=(erows.length)+headerRow;
  headerRow=Math.max(0,headerRow); headerRow=Math.min(headerRow,erows.length-1);

  let erow1=jQuery(erows[headerRow]);
  let  eTds=erow1.find('th,td');
  if (eTds.length==0)  return '-1 No TH or TD  elements in headerRow ('+headerRow+') of id = '+aidSay;

  return eTds;
}        // s_sortTable_init_options2 (within  wsurvey.sortTable.init


//=============
// the options for this column -- the defaults + oprionts + fixups + column speific
 function sortTable_init_options3(etd1,useOpts,customOpts,optsFixems) {

    let optsUseHere={};     // start with defOpts + options

    for (let ie2 in useOpts) optsUseHere[ie2]=useOpts[ie2];  // initialize to the generic values (defops + options)

    for (let ic1=0;ic1<customOpts.length;ic1++) {         // check for column overrides
       let copt=customOpts[ic1];
       var odef=useOpts[copt];          // current "default or options" value -- after various (eg "0" "1") replacements done
       optsUseHere[copt]=wsurvey.sortTable.attr(etd1,copt,odef);     // possible custom replacements? (look for several variants of copt)
    }

    for (let cc1 in  optsFixems) {       // do sokme "0" and "1" substitutions
       if (jQuery.trim(optsUseHere[cc1])=='0') optsUseHere[cc1]='';  // 0 means "don't show"
       if (jQuery.trim(optsUseHere[cc1])=='1') optsUseHere[cc1]= optsFixems[cc1] ; // the "1" defaults
       optsUseHere[cc1] =wsurvey.sortTable.html_entity_decode(optsUseHere[cc1])   ; // fix encoding to that it works in value="xx"  attributes
    }

    return optsUseHere ;
}           // wsurvey.sortTable.init_options3 (within  wsurvey.sortTable.init

//============================
// add a column to the table
   function  sortTable_init_addRowNumbers(rowNumberClass,erows,isOriginal,headerRow,shrinkColIcon) {
   let aa,asort;

   if (isOriginal==0) {
     asort='Table row ';
   } else {
     asort='Original row ';
   }

  for (let ir=0;ir<erows.length;ir++) {
    if (ir==headerRow) {
       if (isOriginal==0) {
          asort='Table row ';
          aa='<td><span class="'+rowNumberClass+'" title="The  table row numbers">&#65283;</span></td>';
       } else {
         asort='Original row ';
         aa='<td  shrinkColIcon="" desIcon="'+shrinkColIcon+'" ascIcon="'+shrinkColIcon+'" des2Icon="" asc2Icon=""  ><span class="'+rowNumberSortableClass+'" title="The  original row numbers"></span></td>';
       }
     } else {
        aa='<td><span class="'+rowNumberClass+'" title="'+asort+'">'+ir+'</span></td>';
     }
     er=jQuery(erows[ir]);
     er.prepend(aa);
  }
  return erows;
 }          // wsurvey.sortTable.init_addRowNumbers (within  wsurvey.sortTable.init

 // --------- end of  swurvey.sortTable.init internal   functions

} //  :::::::::::: wsurvey.sortTable.init  end


//=============
// the event function for "sort button clicks"
// Looks for attributes in the sort button, as they were added by wsurvey.sortTable.init
// Sorting uses the   .text() in  a cell.
// Or, if there is any element in the cell with a _sortUse attribute, the value of this _sortUse attribute is used  (_sortUse can be changed to some other attribute name)
// Sorting can be ascending or descenting; and alphabetic/numeric/or auto (numeric if possible)
// Highlighting of sorted column (or just header) cell is supported
//  Locked cells (values do NOT change after sort) are supported
// Columns with not sort buttons can be specified

wsurvey.sortTable.sortCol=function(athis,arf1,arf2,arf3) {

   var ewas,freezeCols=[]  ;
   var sortColsTodo=[], sortAscTodo=[], sortNumericTodo=[];


  let ethis=wsurvey.argJquery(athis);

// perhaps this is a button outside of the table? In which case, trigger a click on the button inside the table
// (since it uses jquery to find table info for the table that contains the event-causing button)

  let targetButtonUse=ethis.data('targetButtonUse');
  if (typeof(targetButtonUse)!='undefined') {    // this is a "copy of header row" button. So trigger the original
     targetButtonUse.trigger('click');       // click on original button (the one in the table)
     return 0;
  }

// find out info on the table containing this event causing button

  let ethisTd=ethis.closest('td,th');
  let etr=ethisTd.closest('tr');     // the header row
  var etable=etr.closest('table');

// before execution calls.
  let preCallsA=etable.wsurvey_attr('data-wsurveysorttable_precalls','');
 
  let preCalls=preCallsA.split(',');
  for (let afunc0 of preCalls) {
      afunc=jQuery.trim(afunc0);
      if (afunc=='') continue;
      if (typeof(window[afunc])!='function') return false ; // fatal error (but won't e detedted
      let afoo=window[afunc](athis)   ;  // send the event
      if (afoo===false) return false  ;   // false means "no more procesing)
  }

  let ethisTrTds=etr.find('td,th');
  let sortColUse=ethisTrTds.index(ethisTd);

  var  erows=etable.find('tr');   // all the rows!

// get attributes that are table specific
// these are typicall set in  wsurvey.sortTable.init, and will be availble as .data
// . But could be hard coded (say, if  wsurvey.sortTable.init is not being used)-- so check attributes if necdssary

  let sortUseName=etable.data('wsurvey_sortTable_sortUse');
 
  if (typeof(sortUseName)=='undefined') {
      sortUseName=etable.wsurvey_attr('data-wsurveysorttable_wsortuse','_sortUse');
      etable.data('data-wsurveysorttable_sortUse',sortUseName);  // and save for later use
  }

  let startRowUse=etable.data('wsurvey_sortTable_startRowUse');     // will be created by this _init
  if (typeof(startRowUse)=='undefined') {                    // if not, create it from a startRow attribute
     startRowUse=etable.wsurvey_attr('data-wsurveysorttable_startrow',false,1);            // preferred name
     if (startRowUse===false)   startRowUse=etable.wsurvey_attr('data-wsurveysorttable_startrow',1);  // short cut, but not in namespace
     startRowUse=parseInt(startRowUse);
     if (startRowUse>erows.length-1) startRowUse=erows.length-1 ;      // better check!
     etable.data('wsurvey_sortTable_startRowUse',startRowUse);  //and save for later use
  }

  let endRow=etable.data('wsurvey_sortTable_endRowUse');            // same logic as with startRowUse
  if (typeof(endRow)=='undefined') {
      endRow=etable.wsurvey_attr('data-wsurveysorttable_endrow',false,1);
      if (endRow===false)   endRow=etable.wsurvey_attr('data-wsurveysorttable_endrow',erows.length-1);
      if (endRow<0) endRow=erows.length+endRow-1 ;
      endRow=parseInt(endRow);                                            // better check!
      endRow=Math.min(endRow,erows.length-1);
      endRow=Math.max(endRow,startRowUse);  // could highlight one cell
      etable.data('wsurvey_sortTable_endRowUse',endRow);     // and save for later use
   }

  freezeCols=etable.data('wsurvey_sortTable_freezeColsUse');  // if first call to this func, this won't exist
  if (typeof(freezeCols)=='undefined')  {  // not yet created
    freezeCols=[];
    let freezed=etable.wsurvey_attr('data-wsurveysorttable_freezecols',false,1);
    if (freezed===false)   freezed=etable.wsurvey_attr('data-wsurveysorttable_afreeze','');
    if (jQuery.trim(freezed)!='') freezeCols=freezed.split(',');
  }
  etable.data('wsurvey_sortTable_freezeColsUse',freezeCols);  // and save for later use

  let colDescs=etable.data('wsurvey_sortTable_colDescs');         // same logic as with startRowUse
  if (typeof(freezeCols)=='undefined')  {       // not yet created
     let colDescs=[];
     for (let ict=0;ict<ethisTrTds.length;ict++) colDescs[ict]='Col '+ict;
     etable.data('wsurvey_sortTable_colDescs',colDescs);
  }

// done with table speciic attributes. Now check for colum specific ones
//

  let sortSecondary=ethis.wsurvey_attr('data-wsurveysorttable_sortsecondary',0);

  let sortSecondaryTitle= (sortSecondary==1) ? ' (secondary sort)' : ' '; 

// get the "numeric" and "directions" of each of these colums in sortColUse

  let sortNumeric=ethis.wsurvey_attr('data-wsurveysorttable_sortnumeric',-1);  // default is automatic sort

  if (isNaN(sortNumeric)) sortNumeric=-1;
  if (sortNumeric>0) sortNumeric=1;

  let sortAsc=ethis.wsurvey_attr('data-wsurveysorttable_sortasc',1);   // default is ascending sort
  sortAsc= (sortAsc=='1')  ? 1 : 0 ;

//  initialize sortColsTodo , sortAscTodo , sortNumericTodo (info on which columns are sorted, and how). Used by  wsurvey.sortTable.recentSortInfo
  if (sortSecondary!='0') {   // a secondary sort!  If not secondary, used the [] intializations at top of function
    sortColsTodo=etable.data('wsurvey_sortTable_nowSorted');      // what columns are currently sorted 'primary,secondary,...'
    sortAscTodo=etable.data('wsurvey_sortTable_nowAsc');      // what columns are currently sorted 'primary,secondary,...'
    sortNumericTodo=etable.data('wsurvey_sortTable_nowNumeric');      // what columns are currently sorted 'primary,secondary,...'
    if (typeof(sortColsTodo)=='undefined') sortColsTodo=[];       // could happen on first call
    if (typeof(sortAscTodo)=='undefined') sortAscTodo=[];
    if (typeof(sortNumericTodo)=='undefined') sortNumericTodo=[];

    if (sortColsTodo.length==0) sortSecondary=0  ;   // if no primary sort, secondary sort is really a primary sort
  }
 
// see if this is a secondary sort that replaces prior sort on this column
  let sswas=jQuery.inArray(sortColUse,sortColsTodo) ;
  if (sswas>-1) {  // remove this columns sort instructions - reinsert at end (done last)
       sortColsTodo.splice(sswas,1);
       sortAscTodo.splice(sswas,1);
       sortNumericTodo.splice(sswas,1);
  }
  sortColsTodo.push(sortColUse);  // the "current sort button" is last (or first, if nothing already sorted)
  sortAscTodo.push(sortAsc);
  sortNumericTodo.push(sortNumeric);
  let newDirection=1-sortAsc;     // what is the sorting order (of the most secondary) And toggle display to be the other direction
  ethis.attr('data-wsurveysorttable_sortasc',newDirection);
  if (newDirection==1) {
  
     ethis.attr('title','Sort ascending  '+sortSecondaryTitle);
     let aicon=ethis.wsurvey_attr('data-wsurveysorttable_ascicon',false);
     if (aicon!==false) ethis.html(aicon);
     let copyHeaderUsed=ethis.data('copyHeaderButton');
     if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
     if (copyHeaderUsed!==false) {
        copyHeaderUsed.attr('title','Sort ascending '+sortSecondaryTitle);
        let aicon=ethis.wsurvey_attr('data-wsurveysorttable_ascicon',false);
        if (aicon!==false) copyHeaderUsed.html(aicon);
     }
  } else {
     ethis.attr('title','Sort descending '+sortSecondaryTitle);
     let aicon=ethis.wsurvey_attr('data-wsurveysorttable_desicon',false);
     if (aicon!==false) ethis.html(aicon);
     let copyHeaderUsed=ethis.data('copyHeaderButton');
     if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
     if (copyHeaderUsed!==false) {
        copyHeaderUsed.attr('title','Sort descending '+sortSecondaryTitle);
        let aicon=ethis.wsurvey_attr('data-wsurveysorttable_desicon',false);
        if (aicon!==false) copyHeaderUsed.html(aicon);
     }
 }

// done reading specs. Get ready to sort  -- create a matrix of values, that contains a row number, that will be sorted

  if (startRowUse>0) eNonSort=jQuery(erows[startRowUse-1]);  // if start sorting at row >0, start appending trs after this last non sorted row

  let origFreeze=wsurvey.sortTable.sortCol_makeFreeze(erows,freezeCols,startRowUse,endRow) ;

   theVals=wsurvey.sortTable.sortCol_extractVals(erows,sortColsTodo,startRowUse,endRow,sortUseName) ;
   theVals.sort(sortTable_sortCol2);       // and index sort! perhaps numeric? perhaps ascending? uses sortColsTodo, sortAscTodo, sortNumericTodo

// sort -- moving the elements in the collection
  if (startRowUse>0)  eWas=eNonSort ;       // initialize the 'append after last append'
  for (let  i=0;i<theVals.length;i++) {
    let rowUse=startRowUse+i;
    let iReplace=theVals[i][0];
    if (rowUse>0) {                     // not the top row?
       jQuery(eWas).after(erows[iReplace]);      // place after the most recenly place row (or if first time, after the last nonsorting row
       eWas= erows[iReplace] ;
    } else {                       // top row? prepend to the table
       etable.prepend(erows[iReplace])  ;
       eWas= erows[iReplace]
    }
  }   // i=0...


// highlight this col sort icons
 
   let ebuttons=etr.find('[name="wsurvey_sortTable_button"]');
   for (let iz=0;iz<ebuttons.length;iz++) {  // unhighlight all icons
       aetd=jQuery(ebuttons[iz]);
       let eci=wsurvey.sortTable.attr(aetd,'iconcurrentclass','');
       if (eci!='') {
           aetd.removeClass(eci);
           let copyHeaderUsed=aetd.data('copyHeaderButton');
           if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
           if (copyHeaderUsed!==false) copyHeaderUsed.removeClass(eci);
       }

   }
   let iprime=sortColsTodo[0];       // mark primary sort
   let eatdc=ebuttons.filter('[data-wsurveysorttable_colnumber="'+iprime+'"] ');
   if (eatdc.length>0)  eatdc=eatdc.filter('[data-wsurveysorttable_sortsecondary="0"]');
   if (eatdc.length==1) {
         let aci=wsurvey.sortTable.attr(aetd,'iconcurrentclass','');
         if (aci!='') {               // add this class, and note its used for later removal
            eatdc.addClass(aci);
            eatdc.attr('data-wsurveysorttable_currenticonclass_use',aci);
         }
         let copyHeaderUsed=eatdc.data('copyHeaderButton');
        if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
         if (copyHeaderUsed!==false)  copyHeaderUsed.addClass(aci);

   }
   for (let iz2=1;iz2<sortColsTodo.length;iz2++)  {    // mark secondar srots
     let  i2nd=sortColsTodo[iz2];
     let eatdc2=ebuttons.filter('[data-wsurveysorttable_colnumber="'+i2nd+'"] ');
     if (eatdc2.length>0)  eatdc2=eatdc2.filter('[data-wsurveysorttable_sortsecondary="1"]');
     if (eatdc2.length==1) {
        let aci2=eatdc2.wsurvey_attr('data-wsurveysorttable_iconcurrentclass','');
        if (aci2!='') {
          eatdc2.addClass(aci2);
          eatdc2.attr('data-wsurveysorttable_currenticonclass_use',aci2);
         let copyHeaderUsed=eatdc2.data('copyHeaderButton');
         if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
          if (copyHeaderUsed!==false)  copyHeaderUsed.addClass(aci2);

       }
     }
   }    //  for sweoncary

    etable.data('wsurvey_sortTable_nowSorted',sortColsTodo);      // what columns are currently sorted 'primary,secondary,...'
    etable.data('wsurvey_sortTable_nowAsc',sortAscTodo);          // asc/des status of this sort
   etable.data('wsurvey_sortTable_nowNumeric',sortNumericTodo);      // numeric status


  if (origFreeze.length>0)  {  // frozen columns ? WRite their contents to this now sorted table

     let eTrsNew=etable.find('tr');   // read the "now sorted" table

     for (let irr=startRowUse;irr<origFreeze.length;irr++)   { // origfreeze starts at startRowUse
       let eRepTd=jQuery(eTrsNew[irr]).find('td,th');
       for (let ak2 in origFreeze[irr]) {
          let ak2a=parseInt(ak2);
          let tdIn=jQuery(eRepTd[ak2]);
          let afreeze1=origFreeze[irr][ak2] ;   // note that a "clone" is being copied to this location
          tdIn.after(afreeze1);     // place clonse after this
          tdIn.detach();            // and detach the cell (perhapd delete?
       }
    }
  }    // freezecols

// post calls?
  let postCallsA=etable.wsurvey_attr('data-wsurveysorttable_postcalls','');
  let postCalls=postCallsA.split(',');

  for (let afunc0 of postCalls) {
      afunc=jQuery.trim(afunc0);
      if (afunc=='') continue;
      if (typeof(window[afunc])!='function') return false ; // fatal error (but won't e detedted
      let afoo=window[afunc](athis)   ;  // send the event
      if (afoo===false) break  ;   // false means "no more procesing)
  }



  return 1;


 //============     title
 // internal function (can access direction and type instructonsfs
 // sort n>1 col array on 2nd, 3rd,... columns (1st col is the original row == used in index sorts)
 // uses  sortAscTodo, sortNumericTodo

    function sortTable_sortCol2(a,b,ilevel ) {
     
// multi level sort. Sort on [1]. IF tie  sort on [2], if tie sort on[3] ....
    if (arguments.length<3) ilevel=1;          // first call

    let doSortNumeric=sortNumericTodo[ilevel-1];
    let doSortAsc=sortAscTodo[ilevel-1];

    let a1=a[ilevel];       // since all rows start with the "index", sortable values start at [1]
    let b1=b[ilevel];

    if (doSortNumeric==1) {          // fix up values before comparing
        a1=parseFloat(a1);
        b1=parseFloat(b1);
    } else if (doSortNumeric==-1)  {   // sort numeric if both are numbers
       if (!isNaN(a1) && !isNaN(b1)) {
          a1=parseFloat(a1);
          b1=parseFloat(b1);
       }
    }

    if (a1==b1) {
       if (a.length-1<=ilevel) return 0;  // a tie
       let igoo=sortTable_sortCol2(a,b,ilevel+1);  // recurse
       return igoo;
    }
    if (doSortAsc==1) {
        if (a1<b1) return -1 ;
        return 1;
    } else {
        if (a1<b1) return 1 ;
        return -1;
    }
  }


}       ; // wsurvey.sortTable.sortCol

//==================
// helper function (but not internal)
// create a sparse matrix of "frozen" results: for rows startRowUse to endRow, and cols in freezeCols
//  each cell (jrow,jcol) will contain a clone of the cell at the erow[jrow][jcol] location in
// Note that first row of origFreeze has index of startRowUse (does NOT start at 0)

 wsurvey.sortTable.sortCol_makeFreeze=function(erows,freezeCols,startRowUse,endRow) {
  let origFreeze=[] ;    // each row corresponds to row in erows
  if (freezeCols.length==0) return origFreeze;

  for (let iat=startRowUse;iat<=endRow;iat++) {      // for each row subject to sorting
    origFreeze[iat]=[];
    let erowUse=jQuery(erows[iat]);
    let eTds=erowUse.find('th,td');
    for (let io=0;io<freezeCols.length;io++) {
          let io2=freezeCols[io];
          origFreeze[iat][io2]=jQuery(eTds[io2]).clone(true);  // clone into this cell of this "sparse row"
    }
  }
  return origFreeze ;
}


//==============
// extract from erows: from  rows startRowUse to endRow; columns in sortColsTodo
// returns an arrya for each of thse rows
//   [0] : the row number (useful for sorting)
//  [1] : sortCol[0] column
//  ....
// The value returjed depends on sortUseName   (default, or if '', sortUseName= '_sortUse')
//  sortUseName='0'  : extract the .text() of the cell
//  sortUseName='1'  : extract the .html() of the cell
//  sortUseName=sortUseVar   : the value of  the sortUseVar (by default '_sortUse') attribute  from a cell's<td> (or th),
//                       of if not specified in this <td> (or th) from first element within the td (or th) with such an attribute
//                       If no such attribute anywhere, use .text()
//
// note that the one arg shortcut requires that the identified table be pre procssed by wsurvey.sortTable.init

 wsurvey.sortTable.sortCol_extractVals=function(erows,sortColsTodo,startRowUse,endRow,sortUseName,defVal) {

  let  theVals=[];
  if (arguments.length<5) sortUseName='_sortUse';
  if (arguments.length<6) defVal=false;
  if (typeof(sortColsTodo)=='string') {
      if (jQuery.trim(sortColsTodo)==='') {
        sortColsTodo=[];
      } else  {
         sortColsTodo=sortColsTodo.split(',');
      }
  }
  if (typeof(sortColsTodo)=='number') sortColsTodo=[sortColsTodo];

  sortUseName=jQuery.trim(sortUseName);
  if (sortUseName==='')  sortUseName='_sortUse';

// and now copy all values (of sort seelcted columns) to a matrix  (for the rows in range)
  for (let iat=startRowUse;iat<=endRow;iat++) {      // for each row subject to sorting
    let erowUse=jQuery(erows[iat]);
    let eTds=erowUse.find('th,td');
    if (sortColsTodo.length==0)  {   // all columns (using top row as guide to # of columns
       if (eTds.length==0) return [] ;  // give up
       for (let igoo=0;igoo<eTds.length;igoo++) sortColsTodo.push(igoo);

    }
    let sortx='',especial2,especial;
    let svals=[iat];               // if missing column (eg not enough rows), or other problme -- just use row number
    for (let iss0=0;iss0<sortColsTodo.length;iss0++)  {
       let iss=parseInt(sortColsTodo[iss0]);
       sval=(defVal!==false) ? defVal : iat ; // used if not such column
       if (iss<eTds.length){       //  there is such a column! So set sval (using the sortUseName technique)
          let eTdSort=jQuery(eTds[iss]);
          if (sortUseName=='0') {
            sval=eTdSort.text();
          } else    if (sortUseName=='1') {
            sval=eTdSort.html();
          } else {           // look for sortUseVar attribute

            svalTmp=eTdSort.text();
            especial=wsurvey.sortTable.attr(eTdSort,sortUseName,false);   // is there a "_sortUse" attribute (to use when sorting) in the <td>

            if (especial!==false) {
               sval=especial;               // use _sortUse value in the <Td>

            } else {         // no sortUse in td... maybe within an element in this td?

               sortx='data-wsurvey_sortTable_'+sortUseName;
               especial2=eTdSort.find('['+sortx+']');   // preferred (but messy)
               if (especial2.length==0) {
                   sortx='data-'+sortUseName;
                    especial2=eTdSort.find('['+sortx+']');   // preferred (but messy)
               }
               if (especial2.length==0) {
                    sortx= sortUseName;
                    especial2=eTdSort.find('['+sortUseName+']');         // shortcut
               }

               if (especial2.length>0) {
                   sval=wsurvey.sortTable.attr(especial2[0],sortx,sval) ;    // from first element with this attriure
                 sval=jQuery(especial2[0]).wsurvey_attr(sortx,sval,1) ;    // from first element with this attriure
               } else {
                 sval=eTdSort.text();        // no _sortUSe - use the .text() value (
               }     // especial2.length
            }   // especial
        }       // svals either from text, html, or _sortUse or .text()
      }        //  column exists
      svals.push(sval);
    }
    theVals.push(svals);
  }               // buildign list of values to sort on
  return theVals;
}

//=--------------------------------------
// highlight header column by sort order.
// colorlist is an array of colors. If not specified, 3 colors (lightblue, light cyan, lightlime) are used.
// if Colorlist =[],   unhiglight
// if more sort levels than colors, repeat the last one

 wsurvey.sortTable.highlight=function(aid,colorList) {
  if (arguments.length<2 || typeof(colorList)!='object' ) {
    colorList=['048efa','cyan','04fab7'];    //    ['04e6fa','16f5f5','22f5f9','a3f1f3','b3f6f6 ','c6f9f9 '];
  }
  let lastcolor=colorList[colorList.length-1];

  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);

  let headerRow=etable.data('wsurvey_sortTable_headerrow');
  if (typeof(headerRow)=='undefined') headerRow=0;

  let sortColsTodo= etable.data('wsurvey_sortTable_nowSorted');      // what columns are currently sorted 'primary,secondary,...'
  if (typeof(sortColsTodo)=='undefined' || typeof(sortColsTodo)!='object' || sortColsTodo.length==0 ) return 'Table not sorted';

//  var  erowX=etable.find('tr').first() ;
  var  erowX=etable.find('tr') ;
  if (erowX.length==0) return 'No TR in table ';

  for (let iee1=0;iee1<erowX.length;iee1++) {
//    var erow1=jQuery(erowX[headerRow]);
      var erow1=jQuery(erowX[iee1]);

     let  eTds=erow1.find('th,td');
     eTds.attr('bgcolor','');
     for (let i1=0;i1<sortColsTodo.length;i1++) {
       let i2=sortColsTodo[i1];
       eTd1=jQuery(eTds[i2]);
       let usecolor= (i1<colorList.length) ? colorList[i1] : lastcolor ;
       eTd1.attr('bgcolor',usecolor);
    }
  }
  return 'Set header color for '+sortColsTodo.length;
}



//=--------------------------------------
// retrive sort status of a table  (as an array of html strings)
 wsurvey.sortTable.status=function(aid,asData) {
  let amess=[];
  if (arguments.length<2) asData=0;

  let dataX={};

  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);
  let sortColsTodo= etable.data('wsurvey_sortTable_nowSorted');      // what columns are currently sorted 'primary,secondary,...'
  if (typeof(sortColsTodo)=='undefined') sortColsTodo=[];
  dataX['sortList']=sortColsTodo;

  let colDescs=etable.data('wsurvey_sortTable_colDescs');
  if (typeof(colDescs)=='undefined') colDescs=[];
  dataX['colDescs']=colDescs;

  let colWidths=etable.data('wsurvey_sortTable_colWidths');
  if (typeof(colWidths)=='undefined') colWidths=[];
  dataX['colWidths']=colWidths;

  let colScales=etable.data('wsurvey_sortTable_colScales');
  if (typeof(colScales)=='undefined') colScales=[];
  dataX['colScales']=colScales;

  let colWidthsPct=etable.data('wsurvey_sortTable_colWidthsPct');
  if (typeof(colWidthsPct)=='undefined') colWidthsPct=[];
  dataX['colWidthsPct']=colWidthsPct;

  let origTableWidth=etable.data('wsurvey_sortTable_tableWidth' );
  dataX['origTableWidth']=origTableWidth;

// these will be undefined if sortColsTodo is undefined.
  let sortAscTodo=   etable.data('wsurvey_sortTable_nowAsc');          // asc/des status of this sort
  if (typeof(sortAscTodo)=='undefined') sortAscTodo=[];
  dataX['sortAscending']=sortAscTodo ;

  let sortNumericTodo=  etable.data('wsurvey_sortTable_nowNumeric');      // numeric status
  if (typeof(sortNumericTodo)=='undefined') sortNumericTodo=[];
  dataX['sortNumeric']=sortNumericTodo ;

  let  freezeColUse  =  etable.data('wsurvey_sortTable_freezeColsUse');   // which are frozen
  if (typeof(freezeColUse)=='undefined') freezeColUse=[];
  dataX['freezeCols']=freezeColUse ;

  let startRow= etable.data('wsurvey_sortTable_startRowUse');
  dataX['startRow']=startRow;
  let endRow= etable.data('wsurvey_sortTable_endRowUse');
  dataX['endRow']=endRow;

 let sortUse=etable.data('wsurvey_sortTable_sortUse');
  dataX['sortUse']=sortUse;


  if (asData==1) return dataX ;
  let sortAscTodoSay=[];
  for (let iss=0;iss<sortAscTodo.length;iss++) {
    let a1=sortAscTodo[iss];
    let say1= (a1==0)  ? 'descending ' : 'ascending ';
    sortAscTodoSay[iss]=say1;
  }
  sortNumericTodoSay=[];
  for (let iss=0;iss<sortNumericTodo.length;iss++) {
    let a1=sortNumericTodo[iss];
    if (a1==1) {
      say1='Numeric';
    } else if (a1==-1) {
      say1='Automatic';
    } else {
      say1='Alphabetic ';
    }
    sortNumericTodoSay[iss]=say1;
  }

  let ebuttons=etable.find('[name="wsurvey_sortTable_button"]').first(); // the row with at least one sorting button (if sort all rows, could be anywhere
  let etr=ebuttons.closest('tr'); // its row
  let etds=etr.find('th,td');  // all td,th in this row
  sortColsTodoSay=[];
  for (let iee=0;iee<sortColsTodo.length;iee++) {
     let icol=sortColsTodo[iee];
     let etd1=jQuery(etds[icol]);
     let abg=etd1.wsurvey_attr('bgcolor','');
     let cname=jQuery.trim(colDescs[icol]);
     sortColsTodoSay.push('<span style="border:1px solid gray;margin:4px 3px 4px 6px;padding:5px;background-color:'+abg+'" title="column number '+icol+'">'+cname+'</span> ');
  }
  amess.push('Sort rows '+startRow+' to '+endRow);
  amess.push('Sorted columns (primary, secondary,..): '+sortColsTodoSay.join('  '));
  amess.push('Sorted order  (primary, secondary,..): '+sortAscTodoSay.join(' | '));
  amess.push('Sorted type  (primary, secondary,..): '+sortNumericTodoSay.join(' | '));

  if (typeof(freezeColUse)!='undefined' & freezeColUse.length>0) {
     amess.push('Frozen columns: '+ freezeColUse.join(' | '));
  }

  return amess;
}

//================
// set alternating background colors in a table's rows
// rowColors is an array of "alternating" row colors.
// rowColors[0] is the color of the first (header) row.
//   1,2,3 : a series of colors. They will alternate


 wsurvey.sortTable.rowColors=function(aid,rowColors) {
  var etable,sayaid='n.a.', st2;
 if (arguments.length<2)   rowColors=['#dfdfdf','lime','cyan','yellow' ];
 
 if (jQuery.trim(rowColors)=='2') { // special value
    rowColors=['#dce5f2;','#caede1;','#e0f0ce;'];
 }
 if (typeof(rowColors)=='string')  rowColors=rowColors.split(',');


 if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     sayaid=aid;
     if (aid.substr(0,1)!='#') aid="#"+aid;
     etable=jQuery(aid);
 }  else {
     etable=wsurvey.argJquery(aid);
 }
 

 let st1=etable.wsurvey_attr('data-wsurveysorttable_startrow',false,1);
 
 if (st1===false) {       // maybe this is from a button in  a table
    etable=etable.closest('table');
    if (etable.length==0) {
        alert('wsurvey.sortTable.rowColors: could not find a suitable parent table for id '+sayaid);
        return false ;
    }
    let st3=etable.wsurvey_attr('data-wsurveysorttable_startrow',false,1);
    if (st3===false) {
        alert('wsurvey.sortTable.rowColors: could not find a wsurvey.sortTable.init`ialized table for id '+sayaid);
        return false ;
    }
 }


  let oldclass=etable.wsurvey_attr('data-wsurveysorttable_rowclass',false,1);   // use an existing class name if it exsits
  if (oldclass===false) {
     let rr=Math.random();
     let foo=jQuery.trim(rr).substr(3,8);
     oldclass='wsurvey_sortTable_class_'+foo;
     etable.attr('data-wsurveysorttable_rowclass',oldclass);
     etable.addClass(oldclass);
  }


//.stable tr:nth-child(1) {  background-color: gray !important; }
//.stable tr:nth-child(2n+2) {  background-color: lime !important; }
//.stable tr:nth-child(2n+3) {background-color: cyan !important;
// }

  let darules={};
  let asay=oldclass+' tr:nth-child(1)';
  let ccolor=jQuery.trim(rowColors[0]);
  darules[asay]={'background-color':ccolor};
//  for (let ic1=rowColors.length-1;ic1>=1;ic1--) {
 
  for (let ic1=1; ic1<rowColors.length ;ic1++) {

       iadd=ic1+1;
       let asay=oldclass+' tr:nth-child(2n+ '+iadd+')' ;
       let ccolor=jQuery.trim(rowColors[ic1]);
       darules[asay]={'background-color':ccolor};
  }

  wsurvey.addCssRule('wsurvey_sortTable_rowCss',darules);
}


//===================
// return a string of descriptoin information on what this sort button (that was clicked) did (or would) do
// if a 2nd argument is provided, it should be an id (or a dom/jquery container) -- the message is .html() to it
  wsurvey.sortTable.recentSortInfo=function(athis,aid) {
  var amess='';
  let ethis=wsurvey.argJquery(athis);

  let mycol=parseInt(ethis.attr('data-wsurveysorttable_colnumber'));
  let issecondary=ethis.attr('data-wsurveysorttable_sortsecondary');
  let saySecondary= (issecondary==1) ? 'Secondary sort ' : "Sort ";

  let etable=ethis.closest('table');
  let   sortColsTodo=etable.data('wsurvey_sortTable_nowSorted');      // what columns are currently sorted 'primary,secondary,...'
  let   sortAscTodo=etable.data('wsurvey_sortTable_nowAsc');      // what columns are currently sorted 'primary,secondary,...'
  let   sortNumericTodo=etable.data('wsurvey_sortTable_nowNumeric');      // what columns are currently sorted 'primary,secondary,...'

  let   iat=jQuery.inArray(mycol,sortColsTodo);

  if (iat<0) {
    amess="Error: no match with sortColsTodo "+ mycol; // should never happen
  } else {
     let      ascSort=sortAscTodo[iat];
     let      ascSortSay= (ascSort=='1') ? 'Ascending ' : 'Descending ';
     let      numSort=sortNumericTodo[iat];
     let numSortSay='Automatic ';
     if (numSort==0) numSortSay='Alphabetic ';
     if (numSort==1) numSortSay='Numeric  ';
     amess=saySecondary+'  ('+ascSortSay+'/'+numSortSay+') of column '+mycol;
  }

  if (arguments.length<2) return amess;
   if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   eshow=jQuery(aid);

  eshow.html(amess)
  eshow.show();
  return amess
}


//=============================
// copy the header row to a container not in the table (i.e.; something that is always visible
// aid: the id of the table to take the "header row" of (must of been processed with wsurvey.sortTable.init)
// targetId : id of container to write this "header row" to. I
//      If not specified: or if 0 (or false), just return the dom object containing the copied headers (don't attach to window)
// targetAddClass:  classes to add to the "1 row header table". IF not specified, or false, use the aid table's classes
 wsurvey.sortTable.copyHeaderRow=function(aid,targetId,targetAddClass) {
    let classes;
  if (arguments.length<3) targetAddClass=false;
  if (typeof(targetAddClass)!='string') targetAddClass=false;

  jQuery('#'+targetId).show();
  var aidsay='n.a.',etarget=[];
  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     aidsay=aid;
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);

  if (etable.length==0) alert('Error in  wsurvey.sortTable.copyHeaderRow: no such table id '+aidsay);
  let wasborder=etable.wsurvey_attr('border',0);
  if (targetAddClass===false) {
       classes=etable.wsurvey_attr('class','');
  } else {
      classes=targetAddClass;
  }
  

  let colDescs=etable.data('wsurvey_sortTable_colDescs');
  if (typeof(colDescs)=='undefined') colDescs=[];
  let colWidths=etable.data('wsurvey_sortTable_colWidths');
  let colWidthsPct=etable.data('wsurvey_sortTable_colWidthsPct');

  if (arguments.length>1 && targetId!==false ) {
    if ( typeof(targetId)=='string') {
       targetId=jQuery.trim(targetId);
       if (targetId.substr(0,1)!='#') targetId="#"+targetId;
       etarget=jQuery(targetId);
     }
  }
  var lenEtarget=etarget.length;

  let hdrRow=etable.data('wsurvey_sortTable_headerrow');     // the header row  #
 
  if (typeof(hdrRow)=='undefined') {
      if (lenEtarget==0) return 'Error: no wsurvey.sortTable.headerrow infomation in '+aidsay ;
      etarget.html('Error: no wsurvey.sortTable.headerrow infomation in '+aidsay);
      return 1;
   }

   let rd1=''+Math.random();
   let rname0=rd1.substr(3,8);
   let rname='headerCopyTable_'+rname0;

   let etrs=etable.find('tr') ;
   let etr1=jQuery(etrs[hdrRow]);                 // get the header row (often, but not always,0)
   etr1_buttons_lookup=[];
   var etr1_buttons=etr1.find('[name="wsurvey_sortTable_button"]');  // modify the buttons  in this "cloned" header row
   for (var iwtf=0;iwtf<etr1_buttons.length;iwtf++) {  // lookup table for original buttons
        let aetr1=jQuery(etr1_buttons[iwtf]) ;
        let m1=aetr1.attr('data-wsurveysorttable_madebutton');
        etr1_buttons_lookup[m1]=aetr1  ;
   }

   let etr1Clone= jQuery(etr1).clone();      // this is the action! cloned header row (so retains stuff in it)

   let t1='<table  title="A Non moving header row!" id="'+rname+'" ';   // add 1 row table (the row is clone of headers) to the target id
   if (wasborder!=0) t1+= ' border="'+wasborder+'" ';
   if (classes!='') t1+='  class="'+classes+'" ';
   t1+=' > </table> ';
    if (lenEtarget==0) {    // return the dom (don't attach to window)
      etarget=jQuery(t1);
      etarget.append(etr1Clone);
   } else {
      etarget.append(t1);               // attach to the nonmoving container
      let et1=jQuery('#'+rname);
      et1.append(etr1Clone);
   }

// now clean it up a bit
   let allbuttons=etarget.find('[name="wsurvey_sortTable_button"]');  // modify the buttons  in this "cloned" header row

  let mytds=[];
  for (let irr=0;irr<allbuttons.length;irr++) {   // fix display
    let arr=jQuery(allbuttons[irr]);
    let colnum=arr.attr('data-wsurveysorttable_colnumber');
    if (typeof(mytds[colnum])!='undefined') continue ; // got it, so don't do agaihn
    let mytd=arr.closest("th,td");
    mytds[colnum]=mytd;

    let acoltext = (typeof(colDescs[colnum])!='undefined') ? colDescs[colnum] : 'Col '+colnum;
    if (typeof(colWidths[colnum])!=='undefined') {
        let awidth=colWidths[colnum] ;
       let awidthPct=parseInt(colWidthsPct[colnum])+'%' ;
        mytd.html('<span style="display:inline-block;font-size:90%;overflow:hidden;overflow-text:ellipsis;height:1em;color:blue">'+acoltext+'</span>');
        mytd.width(awidth);         //  replaces    mytd.attr('width',colWidths[colnum]);
    }
  }

//  for (let irr=0;irr<allbuttons.length;irr++) { // fix buttons
  for (let irr=allbuttons.length-1;irr>=0;irr--) { // fix buttons
    let arr=jQuery(allbuttons[irr]);
    let colnum=arr.attr('data-wsurveysorttable_colnumber');
    let mytd=mytds[colnum];
    mytd.prepend(arr);

    let sortNumeric0=arr.attr('data-wsurveysorttable_sortnumeric');
    let sortSecondary=arr.attr('data-wsurveysorttable_sortsecondary');

    let sortNumeric='an automatic';
    if (sortNumeric0==0) sortNumeric='an alphabetic ';
    if (sortNumeric0==1) sortNumeric='a numeric';

// note: it would take some work to determine which direction the "original" button (that clicking on this button triggers) would cause
// so don't bother for now (24 march 2021)

    if (sortSecondary==0) {
      arr.attr('Title','Invoke '+sortNumeric+' sort for this column ');
      t1=wsurvey.sortTable.html_entity_decode('&#8597;');
      arr.val(t1);
    } else {
      arr.attr('Title','Invoke secondary  '+sortNumeric+' sort for this column ');
      t1=wsurvey.sortTable.html_entity_decode('&#10577;');
      arr.val(t1);
    }
    arr.attr('name','wsurvey_sortTable_button_alt');
    arr.css({'opacity':0.6});
  }

  let removeEms=['iconCurrentClass','sortasc','ascicon','desicon']; // clear out attributes that are NOT used
  for (let arem in removeEms) allbuttons.removeAttr(arem);
  allbuttons.data('forTable',etable);   // the jquery object of the actual table to be sorted

// and do it again!
  let newAll= jQuery('#'+rname).find('[name="wsurvey_sortTable_button_alt"]');

  for (inn=0;inn<newAll.length;inn++) {
     let  newAllA=jQuery(newAll[inn]) ;
     let m2=newAllA.attr('data-wsurveysorttable_madebutton');
     newAllA.on('click',wsurvey.sortTable.sortCol);
     newAllA.attr('data-wsurveysorttable_fromcopyheader',rname);
     let eorig=jQuery(etr1_buttons_lookup[m2]);
     newAllA.data('targetButtonUse',eorig);     // the original (to be triggered)
     eorig.data('copyHeaderButton',newAllA);   // inform original button (when triggered) of this "triggering" button
     eorig.attr('data-wsurveysorttable_hascopy',m2);
  }

  if (lenEtarget==0) return etarget ;
  return 1;

}

//==========================
// shrink a columum
 wsurvey.sortTable.shrinkCol=function(evt  ) {

   let ethis=wsurvey.argJquery(evt);
   let colNum=ethis.attr('data-wsurveysorttable_colnumber');
   let useClass=evt.data['useclass'];  // wsurvey.sortTable.shrinkClass is always used
   let isShrunk=parseInt(ethis.attr('data-wsurveysorttable_isshrunk'));
   let adesc=ethis.attr('data-wsurveysorttable_desc');
   ethis.attr('data-wsurveysorttable_isshrunk',1-isShrunk);     // change the status flag

   let myTd=ethis.closest('td,th');

   let aButtons=myTd.find('[data-wsurveysorttable_colnumber]') ;  // contains info
   let aButton=jQuery(aButtons[0]);
   let copyHeaderUsed=aButton.data('copyHeaderButton');
    if (typeof(copyHeaderUsed)=='undefined') copyHeaderUsed=false;
   let copyHeaderTd=false;
   if (copyHeaderUsed!==false)  copyHeaderTd=copyHeaderUsed.closest('td,th');     // what is the cell in the 'external header" (if one crated)

   let etable=ethis.closest('[data-wsurveysorttable_init]').first();
   if (etable.length==0) {
      alert("Error in wsurvey.sortTable.shrinkCol: no parent sortable table");
      return 1;
   }
   let erows=etable.find("tr");
   for (let ier=0;ier<erows.length;ier++) {     // shrink all the colnumber cells
      let arow=jQuery(erows[ier]);
      let etds=arow.find('th,td');
      let aetd=jQuery(etds[colNum]);
     if (isShrunk==0) {          // not srunken, so srhink
          aetd.addClass(useClass);
          let aw=aetd.width();
          let gotold=aetd.wsurvey_attr('data-wsurveysorttable_oldwidth',false,1);
          if (gotold===false) aetd.attr('data-wsurveysorttable_oldwidth',aw);   // only set on first srhink
          aetd.css({'width':'12px'});
          let foo=aetd.children();
          foo.addClass('wsurvey_sortSurvey_tempHide');
       } else {
          aetd.removeClass(useClass);
          let foo=aetd.children();
          foo.removeClass('wsurvey_sortSurvey_tempHide');
          let aw=aetd.attr('data-wsurveysorttable_oldwidth');
          aetd.css({'width':aw});
       }
   }

   if (copyHeaderTd!==false) {             // and corresponding cell in external header
       copyHeaderTd.toggleClass(useClass);
       if (isShrunk==0) {          // not srunken, so srhink
          copyHeaderTd.addClass(useClass);
          let aw=copyHeaderTd.width();
          let gotold=copyHeaderTd.wsurvey_attr('data-wsurveysorttable_oldwidth',false,1);
          if (gotold===false)   copyHeaderTd.attr('data-wsurveysorttable_oldwidth',aw);
          copyHeaderTd.css({'width':'12px'});
       } else {
          copyHeaderTd.removeClass(useClass);
          let aw=copyHeaderTd.attr('data-wsurveysorttable_oldwidth');
          copyHeaderTd.css({'width':aw});
       }
   }

    if (isShrunk==0) {      // is now shrunk!
      let goob='<span class="wsurvey_sortTable_unShrinkSpan"  style="cursor:col-resize;background-color:#353435;;color:#fafaea;font-size:11px">&#9853;</span>';
      myTd.prepend(goob);
      myTd.attr('title','Click to unshrink column #'+colNum+' ('+adesc+')');
     evt.stopImmediatePropagation();
     aButtons.prop('disabled',true);                 // don't allow buttons to do anything -- March 2021 -- a bit coarse, other buttons might still be alive
       aButtons.hide();            // don't allow buttons to do anything and hide them   - March 2021 -- a bit coarse, other buttons might still be alive
      window.setTimeout(function() {    // a short wait seems useful
         myTd.on('click',{'class':useClass,'colNum':colNum,'buttons':aButtons,'thistd':myTd,'origEvt':evt,'copyHeaderTd':copyHeaderTd},wsurvey.sortTable.unshrinkCol);
      },100);
   }

}
// ===============
// unwrhink a col
 wsurvey.sortTable.unshrinkCol=function(evt) {
  e1=wsurvey.argJquery(evt,'currentTarget');  // make sure to get the <th> to which this handler was attached
  e1.off('click');
  let aclass=evt.data['class'];
  let myTd=evt.data['thistd'];
  let myUnshrinkIcon=myTd.find('.wsurvey_sortTable_unShrinkSpan');
  myUnshrinkIcon.remove();
  let buttons=evt.data['buttons'];
  let origEvt0=evt.data['origEvt'];
  let copyHeaderTd=evt.data['copyHeaderTd'];
  buttons.prop("disabled",false);
  buttons.show();

  wsurvey.sortTable.shrinkCol(origEvt0);        // this will to the work

  evt.stopImmediatePropagation()   ; // just in case

}

//================
// unshrink all shrunken cols
 wsurvey.sortTable.unShrinkAll=function(aid) {
  var aidsay='n.a.'
  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     aidsay=aid;
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);
  let etest=etable.data('wsurvey_sortTable_init');
 ;
  if (typeof(etest)=='undefined') {
     return  'wsurvey.sortTable.unShrinkAll error: '+aidsay+' is not wsurvey.sortTable initialized table ' ;
  }
  let erowHeader=etable.data('wsurvey_sortTable_erowHeader');
 
  let eshrinks=erowHeader.find('.wsurvey_sortTable_unShrinkSpan');   // this is the cute button..  or wsurvey.sortTable.shrinkClass (the th)

  eshrinks.trigger('click');
  return eshrinks.length ;

}

//===========================
// mark "groups" in  sorted columns. level = 0 for prmary sort, 1 for 2ndary, 2=tertiary, etc
 wsurvey.sortTable.markGroups=function(aid,level,colors,thickness) {
  if (arguments.length<2) level=0;

  var aidsay='n.a.'
  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     aidsay=aid;
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);
  let erowHeader=etable.data('wsurvey_sortTable_init');
  if (typeof(erowHeader)=='undefined') {
     return  'wsurvey.sortTable.markGroups error: '+aidsay+' is not wsurvey.sortTable initialized table ' ;
  }

   if (arguments.length<4) thickness=5;
   if (isNaN(thickness) || thickness<1 ) thickNess=4;
   thickness=parseInt(thickness);

//from https://coolors.co/palettes/trending and https://coolors.co/
let colorDefaults={'cyan5':["#07beb8","#3dccc7","#68d8d6","#9ceaef","#c4fff9"],
           'mixed15':["#dad9b7", "#a87334", "#b9cbb9", "#318ca0", "#c5a871", "#b3874e", "#ceb985", "#69a8a2", "#d7c999", "#4d9e9e",  "#9bbcab",  "#85b3a4",  "#e3e5c7",  "#bb965d",  "#ece4c3"],
           'mixed5':["#ffc09f","#ffee93","#fcf5c7","#a0ced9","#adf7b6"],
           'gray5':["#e0e2db","#d2d4c8","#b8bdb5","#889696","#5f7470"],
           'gold10':["#fffae5","#fff6cc","#fff2b2","#ffee99","#ffe97f","#ffe566","#ffe14c","#ffdd32","#ffd819","#ffd400"],
           'mixed10':["#eae4e9","#fff1e6","#fde2e4","#fad2e1","#e2ece9","#bee1e6","#f0efeb","#dfe7fd","#cddafd"],
           'greenpurple10':["#7400b8","#6930c3","#5e60ce","#5390d9","#4ea8de","#48bfe3","#56cfe1","#64dfdf","#72efdd","#80ffdb"],
           'pink10':["#fadde1","#ffc4d6","#ffa6c1","#ff87ab","#ff5d8f","#ff97b7","#ffacc5","#ffcad4","#f4acb7"]
        } ;
   if (arguments.length<3) colors=colorDefaults['mixed15'] ;
  var qdo=true ;

  if (typeof(colors)=='string' && jQuery.trim(colors).toLowerCase()=='none')  {
     qdo=false;
  } else {
     if (typeof(colors)=='string' || typeof(colors)=='number'  ) {
        if (jQuery.trim(colors)=='' || jQuery.trim(colors)=='0') {
            colors=colorDefaults['mixed15']
        } else {
          let lcolors=colors.toLowerCase();
          if (isNaN(lcolors)) {
            if (typeof(colorDefaults[lcolors])!='undefined') {
              colors=colorDefaults[lcolors] ;
            } else {
               colors=colors.split(',');
            }
          } else {      // isNan
             colors=colorDefaults['mixed15'].slice(0,colors);
          }
        }   // colors ==''
     }    // string or nuber... implciti else is to use as its (colors passed as an array)
  }    // qdo

  let erows=etable.find('tr');

  let tdata=wsurvey.sortTable.status(aid,1);

  let sortColsTodo=tdata['sortList'];   // [0] is prime sort!
  if (sortColsTodo.length==0)    return 0 ;  // nothing sorted. So no grouping

   if (level>=sortColsTodo.length) return 0 ;  // no such secondary sort
  let sortCol=sortColsTodo[level];
  let startRow=tdata['startRow'];
  let endRow=tdata['endRow'];
  let sortUseName=tdata['sortUse'];
  //sortUseName=1;
  let defval=false;

  let theVals=wsurvey.sortTable.sortCol_extractVals(erows,sortCol,startRow,endRow,sortUseName,defval)

  if (theVals.length==0) return  ; // nothing sorted, so no troups

  let groupList=[];        // each row is an 3 element array specifying contignous gorups [value,firstRow,#rows]

  let inow=0,wasval=false,ith=-1 ;
  for (let ih in theVals) {
      ith++;
      let aa=theVals[ih];
      let krow=aa[0] ;  // row
      aval=aa[1] ;  // value
      if (ith==0) {    // initializse
        groupList[inow]=[aval,krow,1] ;
        wasval=aval;
        continue;
     }
     if (aval===wasval) {
        groupList[inow][2]++;
    } else {
       inow++;
       groupList[inow]=[aval,krow,1];
       wasval=aval;
    }
  }

 let ncolors=colors.length;
 for (let ig in groupList) {
     let bb=groupList[ig];
     let gval=bb[0], grow1=bb[1],ngrows=bb[2];
     for (let ig2=0;ig2<ngrows;ig2++) {
           let erow1=jQuery(erows[grow1+ig2]);
           let etds=erow1.find('th,td');
           let etd1=jQuery(etds[sortCol]);  // just the cell of this coluomn
           let cgroup=ig%ncolors;
           let acolor=colors[cgroup];

           if (qdo) {
             etd1.css({'background-color':acolor+' !important'}); // background color
             etd1.attr('bgcolor',acolor);
             etd1.css({'border':thickness+'px inset '+acolor}); // cell border
             etd1.attr('title','group # '+ig+ ' w/value='+gval);
           } else {
//             etd1.css({'background-color':acolor+' !important'}); // cell border
//             etd1.attr('bgcolor',acolor);
             etd1.css({'border':''}); // cell border
             etd1.attr('title','group # '+ig+ ' w/value='+gval);
           }

     }       // all the rows in this group
 }     // this group




}
//function rgbDo0(r, g, b){
//    return `rgb(${r}, ${g}, ${b})`;
//}
// convert r,g,b to #xxyyzz equivalent
 wsurvey.sortTable.surveySort_rgbToHex=function(red, green, blue) {      // https://stackoverflow.com/questions/2173229/how-do-i-write-a-rgb-color-value-in-javascript
    red=parseInt(red); green=parseInt(green); blue=parseInt(blue);
    var decColor =0x1000000+ blue + 0x100 * green + 0x10000 *red ;
    return '#'+decColor.toString(16).substr(1);
}

//======================
// extract an attribute set by survey.sortTable (From jquer object athis
// athis: element to look for attribute. Usuually jquery object, but could be #aid
// anattr: attribute name to look for.
// adef: defatult value if attribute not found. If not specified, false is used
// newval: is speified, don't look for attribute -- set it. The adef is ignored if newval is set
// When looking for an attribute, theo rder is:
//  i) look for data-wsurveysortable_anattr. If no exact (but case insenstive) match found
//  ii) look for data-anattr. If not exact (but case insenstive) match found
//  iii) look for anattr.  If no exact (but case insensitive) match found, return adev
// When setting an attribute, data-wsurveysorttable_anattr is used
// If aid is NOT a jquery (or jquery possible) identifier, return defval

 wsurvey.sortTable.attr=function(athis,anattr,adef,newval) {
  if (arguments.length<2) adef=false ;   // default if not specified

  var dname,vv;

  let ethis=jQuery(athis).first()  ; // usually unnecessary
  if (ethis.length==0) return adef;

  daname='data-wsurveysorttable_'+anattr ;

  if (arguments.length==4)  {              // set the attribute
    ethis.attr(daname,newval);
    return 1;
  }

// not set .. look for ...
 
  vv=ethis.wsurvey_attr(daname,false,1);
  if (vv!==false) return vv;              // match to prefered form: data-wsurveysorttable_anattr

  daname='data-'+anattr ;
 ;
  vv=ethis.wsurvey_attr(daname,false,1);
  if (vv!==false) {
   // alert(daname+ ' got '+vv);
     return vv;              // 2nd prefered to prefered form: data-anattr
  }


  vv=ethis.wsurvey_attr(anattr,adef,1);
 if (vv!==false) return vv;              //  as  anattr

  return adef ;       // no match
}

//=============
// assign width to all table cells
// If ascale0 is a scalar, all widths are adjusted using it alue
// If an array, the corresponding element is used. It the element is NOT present, no adjustment (i.e.; it is NOT rescaled to orignal size)
// I.e  aScale0=[]; ascale0[3]=1.2; ascale0[0]=0.9 means "col set at 90% of original, col 3 as 120%

 wsurvey.sortTable.setColWidths=function(aid,aScale0) {
  if (arguments.length<2) aScale0=1.1;

  var aidsay='n.a.'
  if (typeof(aid)=='string') {
     aid=jQuery.trim(aid);
     aidsay=aid;
     if (aid.substr(0,1)!='#') aid="#"+aid;
  }
  let   etable=jQuery(aid);
  let erowHeader=etable.data('wsurvey_sortTable_init');
  if (typeof(erowHeader)=='undefined') {
     return  'wsurvey.sortTable.assignWidth error: '+aidsay+' is not wsurvey.sortTable initialized table ' ;
  }


  let tstat=wsurvey.sortTable.status(aid,1);

  let erows=etable.find('tr');
  let cwidths=tstat['colWidths'];

  scaleUse=[];
  for (let iss=0;iss<cwidths.length;iss++)   scaleUse.push(1.0);  // default is reset all columns

  if (typeof(aScale0)=='number') {        // scale all columns by same amount
     if (aScale0<=0)   'wsurvey.sortTable.assignWidth error: bad scaling factor ('+aScale0+')';
     for (let iss=0;iss<cwidths.length;iss++)    scaleUse[iss]=ascale;
  }     // number

  if (typeof(aScale0)=='string') {   // all by same amount, or csv of 0,1,.. columns
     let arf=aScale0.split(',');
     if (arf.length==1) {           // not a csv
         let bscale=parseFloat(aScale0);
         if (bscale<=0)   'wsurvey.sortTable.assignWidth error: bad scaling factor ('+aScale0+')';
         for (let iss=0;iss<scaleUse.length;iss++)    scaleUse[iss]=bscale;

     }   else {           // acsv

         for (let iss1=0;iss1<arf.length;iss1++)  {      // if not enough, use default of 1.0
            if (iss1>=scaleUse.length) break ;   // ignore if too many
            let tx=jQuery.trim(arf[iss1]);
            let tx1= (tx=='')  ? 1.0 : parseFloat(tx);
            if (tx1<=0)   'wsurvey.sortTable.assignWidth error: bad scaling factor ('+aScale0+')';
            scaleUse[iss1]=tx1 ;
         } //csv
     }   // not csv
  }    // type of string

  if (typeof(aScale0)=='object') {     // a "sparse" array
    for (let ass in aScale0) {
      iass=ass;
      if (isNaN(ass)) continue ;
      iass=parseInt(ass);
      if (iass<0 || iass > cwidths.length) continue;
      let vscale=parseFloat(aScale0[iass]);
      if (vscale==0)   'wsurvey.sortTable.assignWidth error: bad scaling factor ('+aScale0[iass] +')';
      scaleUse[iass]=vscale;
    }
  }

  for (let ir=0;ir<erows.length;ir++) {   // increase by 10%
      let etr=jQuery(erows[ir]);
      let etds=etr.find('td,th');
      for (let ietd=0;ietd<scaleUse.length;ietd++) {
          let aetd=jQuery(etds[ietd]);
          let thisWidth=cwidths[ietd];
          let bscale=scaleUse[ietd];
          thisWidth=parseInt(thisWidth*bscale);
          let oof={'width':thisWidth};
          aetd.css(oof);
      }   // ietd
  }  // ir

   etable.data('wsurvey_sortTable_colScales',scaleUse);
}

// ============
// utility functions
 wsurvey.sortTable.html_entity_decode = function(str) {   //https://stackoverflow.com/questions/43987539/using-element-val-with-html-entities
    return str.replace(/&#(x[0-9a-fA-F]+|\d+);/g, function(match, dec) {
      return String.fromCharCode(dec.substr(0, 1) == 'x' ? parseInt(dec.substr(1), 16) : dec);
    })
  }  // end of wsurvey.sortTable.html_entity_decode

